using System;
using System.Collections;
using System.Reflection;
using System.Drawing;

using Microsoft.DirectX;
using Microsoft.DirectX.DirectInput;

using DarkStrideToolbox;

namespace DarkStride.StellarLanes.SharedDLL
{
	public class ComplexEntity : Entity
	{
		#region Properties
		private string m_sName = string.Empty;
		private double m_nEnergyForRound = 0;
		public int m_nBaseHUDLevel = (int)enumHUDDetail.Name;

		private DSSortedList m_oChasses = new DSSortedList();
		private Chassis m_oBaseChassis = null;
		#endregion


		public ComplexEntity()
		{
			this.CanPickUpItems = true;
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			Chassis oLoopChassis = null;


			base.Advance( oSession,nElapsedTime );

			m_nEnergyForRound = GetPropertyTotal( enumEntProperties.Energy_Produced ) * nElapsedTime;

			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				oLoopChassis.Advance( oSession,nElapsedTime );
			}
		}
		public override void PreRenderRenderSettingsChange( RenderSettings oRenderSettings )
		{
			Chassis oLoopChassis = null;

			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				oLoopChassis.PreRenderRenderSettingsChange( oRenderSettings );
			}

            base.PreRenderRenderSettingsChange(oRenderSettings);
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			Chassis oLoopChassis = null;


            base.Render(oRenderSettings);

			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				oLoopChassis.Render( oRenderSettings );
			}

			if( oRenderSettings.RenderType == enumRenderType.HUD && Globals.Inst().OurShip == this )
			{
				RenderHPBarHUD( oRenderSettings );
			}
		}

		public override Vector2 GetRotatedWorldPt( Vector2 vWorldPt )
		{
			Vector2 vWorldPtCenterBaseChassis = Vector2.Empty;
			Vector2 vWorldPtEndPt = Vector2.Empty;
			double nAngle = 0;
			double nDist = 0;

			
			vWorldPtCenterBaseChassis = m_oBaseChassis.GetCenterWorldPt();

			//10/08/2007 Chris Hill  Ok this may seem odd, but its right.  I just changed it so that - is up, to 
			//be in line with the graphic system of 0,0 in the upper left corner.  Problem is the angle system
			//(aka radians) uses -Y to be down.  So we have to flip it for all angle calculations.
			nAngle = DSMath.CalculateRadAngle( vWorldPtCenterBaseChassis.X,-vWorldPtCenterBaseChassis.Y,vWorldPt.X,-vWorldPt.Y );
			nDist = DSMath.Distance( vWorldPtCenterBaseChassis.X,vWorldPtCenterBaseChassis.Y,vWorldPt.X,vWorldPt.Y );

			vWorldPtEndPt = new Vector2( 
				(float)( Math.Cos( nAngle - base.Angle ) * nDist + vWorldPtCenterBaseChassis.X ),
				(float)( Math.Sin( nAngle - base.Angle ) * -nDist + vWorldPtCenterBaseChassis.Y ) );


			return( vWorldPtEndPt );
		}

		public override void KeyProcess( Session oSession,Microsoft.DirectX.DirectInput.Key oKey, bool bPressed )
		{
			Chassis oLoopChassis = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
                oLoopChassis.KeyProcess(oSession, oKey, bPressed);
			}
		}
        public override void NetworkMessageReceived(Session oSession, long nMessageType, string sMessage)
		{
			Chassis oLoopChassis = null;


            base.NetworkMessageReceived(oSession,nMessageType, sMessage);

			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
                oLoopChassis.NetworkMessageReceived(oSession, nMessageType, sMessage);
			}			
		}

		#region Chassis/Module placement
		public void AddChassis( Chassis oChassis )
		{
			if( m_oChasses.Count == 0 )
			{
				m_oBaseChassis = oChassis;
			}
			oChassis.ParentEntity = this;
			m_oChasses.Add( oChassis.PKey,oChassis );
		}
		public void RemoveChassis( Chassis oChassis )
		{
			if( m_oBaseChassis == oChassis )
			{
				if( m_oChasses.Count == 0 )
				{
					m_oBaseChassis = (Chassis)m_oChasses.GetByIndex( 0 );
				}
				else
				{
					m_oBaseChassis = null;
				}
			}
			oChassis.ParentEntity = null;

			oChassis.UnAssign();
			m_oChasses.Remove( oChassis.PKey );
		}

		public virtual bool CGridIsInUse( int nXOffset,int nYOffset )
		{
			Chassis oLoopChassis = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				if( oLoopChassis.CGridIsInUse( nXOffset - oLoopChassis.XOffset,nYOffset - oLoopChassis.YOffset ) == true )
				{
					return( true );
				}
			}


			return( false );
		}
		
		public Chassis GetChassisAtCGridLoc( int nCXOffset,int nCYOffset )
		{
			Chassis oChassisFound = null;
			Chassis oLoopChassis = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				if( oLoopChassis.CGridIsInUse( nCXOffset - oLoopChassis.XOffset,nCYOffset - oLoopChassis.YOffset ) == true )
				{
					oChassisFound = oLoopChassis;
					break;
				}
			}


			return( oChassisFound );
		}

		public virtual bool CanChassisBePlaced( Chassis oChassis,int nCXOffset,int nCYOffset,ref ArrayList oaConflicts )
		{
			ArrayList oaShipConnectors = null;
			ArrayList oaMyConnectors = null;
			bool bFoundConflict = false;
			bool bFoundConnector = false;
			System.Drawing.Point oConflict = System.Drawing.Point.Empty;
			System.Drawing.Point oMyConnector = System.Drawing.Point.Empty;
			System.Drawing.Point oShipConnector = System.Drawing.Point.Empty;


			oaConflicts = new ArrayList();

			//Is this our new base chassis?
			if( m_oChasses.Count == 0 )
			{
				bFoundConflict = false;
				bFoundConnector = true;
			}
			else
			{
				//Start by checking simple overrights
				for( int nX=0 ; nX<oChassis.CGridWidth ; nX++ )
				{
					for( int nY=0 ; nY>-oChassis.CGridHeight ; nY-- )
					{
						if( oChassis.CGridIsInUse( /*oChassis.XOffset + */nX,/*oChassis.YOffset + */nY ) == true &&
							CGridIsInUse( nCXOffset+nX,nCYOffset+nY ) == true )
						{
							oConflict = new Point( nX,nY );
							oaConflicts.Add( oConflict );
							bFoundConflict = true;
						}
					}
				}

				//Check connectors, but only if there were no conflicts
				if( bFoundConflict == false )
				{
					int nTempX = oChassis.XOffset;
					int nTempY = oChassis.YOffset;
					oChassis.XOffset = nCXOffset;
					oChassis.YOffset = nCYOffset;

					oaMyConnectors = oChassis.GetCGridConnectors();

					oChassis.XOffset = nTempX;
					oChassis.YOffset = nTempY;

					oaShipConnectors = GetCGridConnectors();
					//Now check them
					for( int nMyConIndx=0 ; nMyConIndx<oaMyConnectors.Count && bFoundConnector == false ; nMyConIndx++ )
					{
						oMyConnector = (System.Drawing.Point)oaMyConnectors[ nMyConIndx ];

						for( int nShipConIndx=0 ; nShipConIndx<oaShipConnectors.Count ; nShipConIndx++ )
						{						
							oShipConnector = (System.Drawing.Point)oaShipConnectors[ nShipConIndx ];

							//Are these next to each other?
							if( ( oShipConnector.X >= oMyConnector.X-1 && oShipConnector.X <= oMyConnector.X+1 && oShipConnector.Y == oMyConnector.Y ) ||
								( oShipConnector.Y >= oMyConnector.Y-1 && oShipConnector.Y <= oMyConnector.Y+1 && oShipConnector.X == oMyConnector.X ) )
							{
								bFoundConnector = true;
								break;
							}
						}
					}
				}
			}


			return( bFoundConflict == false && bFoundConnector == true );
		}
		public virtual bool CanModuleBePlaced( Module oModule,int nXOffset,int nYOffset,ref ArrayList oaConflicts,ref Chassis oChassis )
		{
			DSSortedList oChassisThatWork = new DSSortedList();
			DSSortedList oChassisThatWorkThisRound = null;
			bool bFoundConflict = false;
			bool bFoundSpot = false;
			bool bCantPlaceHere = false;
			int nModuleTypeNeeded = 0;
			int nModuleTypeFound = 0;
			System.Drawing.Point oConflict = System.Drawing.Point.Empty;
			Chassis oLoopChassis = null;
            

			oaConflicts = new ArrayList();

			//Is this our new base chassis?
			if( m_oChasses.Count == 0 )
			{
				bFoundConflict = true;
			}
			else
			{
				oChassisThatWork = m_oChasses.Clone();

				//Walk through all the places for the module and check to see if they are in use
				for( int nX=0 ; nX<oModule.MGridWidth ; nX++ )
				{
					for( int nY=0 ; nY>-oModule.MGridHeight ; nY-- )
					{
						nModuleTypeNeeded = (int)oModule.MGridTypeNeeded( nX,nY );

						//Even if found spot is true we have to keep going in case we find a conflict
						if( oModule.MGridIsInUse(nX, nY) == false ||
                            nModuleTypeNeeded != (int)enumModuleType.CantBeUsed )
						{
							//Ok we have a need... see if any chassis will satisfy it.
							bFoundSpot = false;
							bCantPlaceHere = false;
							oChassisThatWorkThisRound = new DSSortedList();
							for( int nChassisIndex=0 ; nChassisIndex<m_oChasses.Count ; nChassisIndex++ )
							{
								oLoopChassis = (Chassis)m_oChasses.GetByIndex( nChassisIndex );

								nModuleTypeFound = (int)oLoopChassis.MGridModuleTypeAllowed( 
									nX + nXOffset - oLoopChassis.XOffset,
									nY + nYOffset - oLoopChassis.YOffset );
								if( ( nModuleTypeNeeded & (int)enumModuleType.OpenSpace ) == (int)enumModuleType.OpenSpace &&
									( nModuleTypeFound & (int)enumModuleType.OpenSpace ) != (int)enumModuleType.OpenSpace )
								{
									//Failed because this isn't space!
									oConflict = new Point( nX,nY );
									oaConflicts.Add( oConflict );
									bFoundConflict = true;
									bCantPlaceHere = true;
									bFoundSpot = false;
									return( bFoundConflict == false );
								}
									//Is this chassis valid to place this module on?
								else if( oChassisThatWork.ContainsKey( oLoopChassis.PKey ) == true &&
									oLoopChassis.MGridIsInUse( nX + nXOffset - oLoopChassis.XOffset,
									nY + nYOffset - oLoopChassis.YOffset ) == false &&
									( nModuleTypeFound & nModuleTypeNeeded ) == nModuleTypeNeeded )
								{
									bFoundSpot = true;
									oChassisThatWorkThisRound.Add( oLoopChassis.PKey,oLoopChassis );
									oChassisThatWork.Remove( oLoopChassis.PKey );
								}
							}

							//Did we satisfy this spot?
							if( ( ( nModuleTypeNeeded & (int)enumModuleType.OpenSpace ) != (int)enumModuleType.OpenSpace || bCantPlaceHere == true ) &&
								bFoundSpot == false )
							{
								oConflict = new Point( nX,nY );
								oaConflicts.Add( oConflict );
								bFoundConflict = true;
							}
							else
							{
								oChassisThatWork = oChassisThatWorkThisRound;
							}
						}
					}
				}
			}

			if( oChassisThatWork.Count > 0 )
			{
				oChassis = (Chassis)oChassisThatWork.GetByIndex( 0 );
			}


			return( bFoundConflict == false );
		}

		public virtual ArrayList GetCGridConnectors()
		{
			System.Drawing.Point oConnector = System.Drawing.Point.Empty;
			ArrayList oaAllConflicts = new ArrayList();
			ArrayList oaChassisConflicts = new ArrayList();
			Chassis oLoopChassis = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				oaChassisConflicts = oLoopChassis.GetCGridConnectors();
				
				for( int nConflictIndex=0 ; nConflictIndex<oaChassisConflicts.Count ; nConflictIndex++ )
				{
					oaAllConflicts.Add( oaChassisConflicts[ nConflictIndex ] );
				}
			}


			return( oaAllConflicts );
		}

		public ArrayList GetMGridPlacableLocs( Module oModule )
		{
			Chassis oFoundChassis = null;
			Chassis oLoopChassis = null;
			ArrayList oaConflicts = new ArrayList();
			ArrayList oaPlacableLocs = new ArrayList();
			System.Drawing.Point oPlacableLoc = System.Drawing.Point.Empty;


			for( int nChassisIdx=0 ; nChassisIdx<m_oChasses.Count ; nChassisIdx++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( nChassisIdx );

				//Make sure you look one full module size on each outer edge of the chassis.  That way if it extends over into
				//space your good to go.
				for( int nX=-oModule.MGridWidth ; nX<oModule.MGridWidth + oLoopChassis.MGridWidth ; nX++ )
				{
					for( int nY=oModule.MGridHeight ; nY>-oModule.MGridHeight - oLoopChassis.MGridHeight ; nY-- )
					{
						oPlacableLoc = new System.Drawing.Point( oLoopChassis.XOffset + nX,oLoopChassis.YOffset + nY );

						if( IsPlacableLocInList( oaPlacableLocs,oPlacableLoc ) == false &&
							CanModuleBePlaced( oModule,
							oLoopChassis.XOffset + nX,
							oLoopChassis.YOffset + nY,
							ref oaConflicts,ref oFoundChassis ) == true )
						{
							oaPlacableLocs.Add( oPlacableLoc );
						}
					}
				}	
			}


			return( oaPlacableLocs );
		}
		private bool IsPlacableLocInList( ArrayList oaPlacableLocs,System.Drawing.Point oPointToFind )
		{
			System.Drawing.Point oLoopPoint = System.Drawing.Point.Empty;

			for( int i=0 ; i<oaPlacableLocs.Count ; i++ )
			{
				oLoopPoint = (System.Drawing.Point)oaPlacableLocs[i];
				if( oLoopPoint.X == oPointToFind.X && oLoopPoint.Y == oPointToFind.Y )
				{
					return( true );
				}
			}

			return( false );
		}


		public DSSortedList GetChassisThatCannotBeRemoved()
		{
			DSSortedList oTempChassisList = null;
			DSSortedList oChassisThatCannotBeRemoved = new DSSortedList();
			Chassis oLoopChassis = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );

				//Get my children
				oTempChassisList = GetDependentChildrenOfChassis( oLoopChassis );
				//If this guy has dependent children he can't be removed
				if( oTempChassisList.Count > 0 )
				{
					oChassisThatCannotBeRemoved.Add( oLoopChassis.PKey,oLoopChassis );
				}
			}


			return( oChassisThatCannotBeRemoved );
		}
		public DSSortedList GetDependentChildrenOfChassis( Chassis oChassisBeingRemoved )
		{
			DSSortedList oChassisToWalkNext = new DSSortedList();
			DSSortedList oChassisToWalkNow = new DSSortedList();
			DSSortedList oTempChassisAttached = new DSSortedList();
			DSSortedList oChassisThatAreChildren = new DSSortedList();
			Chassis oLoopChassisForNow = null;
			Chassis oLoopChassisForNext = null;


			//Get a list of all the potential children of the chassis being removed
			oChassisThatAreChildren = m_oChasses.Clone();
			oChassisThatAreChildren.Remove( m_oBaseChassis.PKey );

			//If I am the base then everything is a dependent child to me... and the algorithm breaks
			if( oChassisBeingRemoved != m_oBaseChassis )
			{
				//Get a list of all the chassis attached to the base chassis without my help
				oChassisToWalkNow = GetChassesAttachedToThisChassis( m_oBaseChassis );

				//Before we even start, some chassis are obviously not children, like this first wave and the base
				for( int nNowIdx=0 ; nNowIdx<oChassisToWalkNow.Count ; nNowIdx++ )
				{
					oLoopChassisForNow = (Chassis)oChassisToWalkNow.GetByIndex( nNowIdx );
					oChassisThatAreChildren.Remove( oLoopChassisForNow.PKey );
				}

				//Ok now start an iterative loop
				while( oChassisToWalkNow.Count > 0 )
				{
					//Now, for every chassis thats attached to the base get its list of chassis
					for( int nNowIdx=0 ; nNowIdx<oChassisToWalkNow.Count ; nNowIdx++ )
					{
						oLoopChassisForNow = (Chassis)oChassisToWalkNow.GetByIndex( nNowIdx );

						if( oLoopChassisForNow != m_oBaseChassis && oLoopChassisForNow != oChassisBeingRemoved )
						{
							//Add the chassis connected to this chassis, into the chassis to check next
							oTempChassisAttached = GetChassesAttachedToThisChassis( oLoopChassisForNow );
							for( int nNextIdx=0 ; nNextIdx<oTempChassisAttached.Count ; nNextIdx++ )
							{
								oLoopChassisForNext = (Chassis)oTempChassisAttached.GetByIndex( nNextIdx );

								if( oChassisThatAreChildren.ContainsKey( oLoopChassisForNext.PKey ) == true &&
									oChassisToWalkNow.ContainsKey( oLoopChassisForNext.PKey ) == false &&
									oChassisToWalkNext.ContainsKey( oLoopChassisForNext.PKey ) == false )
								{
									oChassisToWalkNext.Add( oLoopChassisForNext.PKey,oLoopChassisForNext );
									//This one can't be a child then
									oChassisThatAreChildren.Remove( oLoopChassisForNext.PKey );
								}
							}
						}
					}

					//The list of chassis we have identified as being next to check, check now
					oChassisToWalkNow = oChassisToWalkNext;
					oChassisToWalkNext = new DSSortedList();
				}
			}


			return( oChassisThatAreChildren );
		}
		private DSSortedList GetChassesAttachedToThisChassis( Chassis oChassis )
		{
			DSSortedList oChassesAttached = new DSSortedList();
			Chassis oLoopChassis = null;


			//Walk all the chassis and see if they are directly connected to this one
			for( int nChassisIdx=0 ; nChassisIdx<m_oChasses.Count ; nChassisIdx++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( nChassisIdx );

				if( oLoopChassis != oChassis &&
					IsChassisAttachedToChassis( oChassis,oLoopChassis ) == true )
				{
					oChassesAttached.Add( oLoopChassis.PKey,oLoopChassis );
				}
			}


			return( oChassesAttached );
		}
		private bool IsChassisAttachedToChassis( Chassis oChassisA,Chassis oChassisB )
		{
			System.Drawing.Point oConnectorA = System.Drawing.Point.Empty;
			System.Drawing.Point oConnectorB = System.Drawing.Point.Empty;
			ArrayList oConnectorsA = null;
			ArrayList oConnectorsB = null;
			bool bConnected = false;


			oConnectorsA = oChassisA.GetCGridConnectors();
			oConnectorsB = oChassisB.GetCGridConnectors();

			//A chassis is connected if its connector is next to my connector, so walk all my connectors
			for( int nA=0 ; nA<oConnectorsA.Count ; nA++ )
			{
				oConnectorA = (System.Drawing.Point)oConnectorsA[nA];

				for( int nB=0 ; nB<oConnectorsB.Count ; nB++ )
				{
					oConnectorB = (System.Drawing.Point)oConnectorsB[nB];

					//Now... are these two connectors next to each other?
					if( Math.Abs( oConnectorA.X - oConnectorB.X ) <= 1 &&
						Math.Abs( oConnectorA.Y - oConnectorB.Y ) <= 1 &&
						(
						oConnectorA.X == oConnectorB.X ||
						oConnectorA.Y == oConnectorB.Y
						)
						)
					{
						bConnected = true;
						nA = oConnectorsA.Count;
						nB = oConnectorsB.Count;
					}
				}
			}


			return( bConnected );
		}

		#endregion

		public override DSSerialize Serialize(Session oSession)
		{
			Chassis oLoopChassis = null;
			DSSerialize oSerialize = new DSSerialize();


			try
			{
				//Now save our data
                oSerialize.Set(0, base.Serialize(oSession));
				oSerialize.SetList( 1 );
			
				for( int i=0 ; i<m_oChasses.Count ; i++ )
				{
					oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );

                    if (oLoopChassis.GetType() != typeof(GenericChassis))
                    {
                        oSerialize.SetListItem(1, oLoopChassis.GetType().Name);
                    }
                    else
                    {
                        oSerialize.SetListItem(1, null);
                    }
                    oSerialize.SetListItem(1, oLoopChassis.Serialize(oSession));
				}

				if( m_oBaseChassis != null )
				{
					oSerialize.Set( 2,m_oBaseChassis.PKey );
				}
				else
				{
					//throw new System.Exception( "No base chassis was found." );
				}

			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			string sTemp = string.Empty;
			Assembly oControlAssembly = null;
			Type oControlType = null;
			ArrayList oChassis = null;
			Chassis oNewChassis = null;
			Chassis oLoopChassis = null;
			DSSortedList oChassisNotUsedYet = null;

 
			oChassisNotUsedYet = m_oChasses.Clone();
			m_oChasses.Clear();


			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );


			oChassis = oSerialize.GetList( 1 );
			//Walk the entitys and update them
			for( int i=0 ; i<oChassis.Count ; i+=2 )
			{
				//Deserialize the form into a new object... so first create the new object
				oControlAssembly = Assembly.Load("StellarLanes");//.LoadWithPartialName( "StellarLanes" );//(string)oChassis[ i+1 ] );
                if (oChassis[i] == null)
                {
                    sTemp = typeof(GenericChassis).Name;
                }
                else
                {
                    sTemp = (string)oChassis[i];
                }
                oControlType = oControlAssembly.GetType("DarkStride.StellarLanes.SharedDLL." + sTemp);
				//Make it!
				oNewChassis = (Chassis)Activator.CreateInstance( oControlType,new object[]{ this } );
				//Now deserialize the form
				oNewChassis.DeSerialize( oSession,(DSSerialize)oChassis[ i+1 ] );


				//Do we have this entity?
				oLoopChassis = (Chassis)oChassisNotUsedYet.GetByKey( oNewChassis.PKey );
				//oLoopChassis = (Chassis)m_oChasses.GetByKey( oNewChassis.PKey );
				if( oLoopChassis == null )
				{
					oNewChassis.ParentEntity = this;
					AddChassis( oNewChassis );
				}

				//If we found it then update it, we can do this outside the lock
				if( oLoopChassis != null )
				{
					oLoopChassis.DeSerialize( oSession,(DSSerialize)oChassis[ i+1 ] );
					AddChassis( oLoopChassis );
					oChassisNotUsedYet.Remove( oLoopChassis.PKey );
				}
			}


			//Find our base chassis
			if( oSerialize.NumberOfParamaters <= 2 )
			{
				//Chris
				//throw new System.Exception( "No base chassis was sent in the serialized stream." );
			}
			if( oSerialize.GetString( 2 ).Length > 0 )
			{
				m_oBaseChassis = (Chassis)m_oChasses.GetByKey( oSerialize.GetString( 2 ) );
			}
			else if( m_oChasses.Count > 0 )
			{
				m_oBaseChassis = (Chassis)m_oChasses.GetByIndex( 0 );
			}
			if( m_oBaseChassis == null )
			{
				for( int i=0 ; i<m_oChasses.Count ; i++ )
				{
					oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
					if( oLoopChassis.GUID_Obsolete == oSerialize.GetString( 2 ) ||
						oLoopChassis.PKey == oSerialize.GetString( 2 ) )
					{
						m_oBaseChassis = oLoopChassis;
						break;
					}
				}
			}
			if( m_oBaseChassis == null && m_oChasses.Count > 0 )
			{
				m_oBaseChassis = (Chassis)m_oChasses.GetByIndex( 0 );
			}


			//All done?  Any spare chassis?  If so they go into unassigned.
			if( Globals.Inst().Profile != null && Globals.Inst().OurShip == this )
			{
				for( int i=0 ; i<oChassisNotUsedYet.Count ; i++ )
				{
					oLoopChassis = (Chassis)oChassisNotUsedYet.GetByIndex( i );
					oLoopChassis.UnAssign();
                    //If its our players ship then put it in the unassigned chassis
                    Globals.Inst().Profile.AddChassis(oLoopChassis);
				}
			}
		}
        
		public override ArrayList GetRegions()
		{
			Chassis oLoopChassis = null;
			ArrayList oRegions = new ArrayList();
			ArrayList oTemp = null;


			//if( this.ImDead == false )
		{
			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );

				oTemp = oLoopChassis.GetRegions();
				for( int n=0 ; n<oTemp.Count ; n++ )
				{
					oRegions.Add( oTemp[n] );
				}
			}
		}


			return( oRegions );
		}
        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
			Chassis oLoopChassis = null;
			ArrayList oRegions = new ArrayList();
			Region oHitTestResults = null;
			DSSortedList oTempList = null;
            int nLiveChassisCount = 0;
			double nTotalStructureRemaining = 0;
			double nTempDamage = 0;
			double nPostDamageRedux = nAmount;
			double nDamageNotUsed = 0;
			bool bMainChassisHitFound = false;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
                if (oLoopChassis.StructurePoints > 0)
                {
                    nPostDamageRedux = oLoopChassis.GetPreArmorDamageReduction(nPostDamageRedux);
                    nLiveChassisCount++;
                }
			}

			//Get a copy of our chassis so that if one is removed we are still ok
			oTempList = m_oChasses.Clone();

			for( int i=0 ; i<oTempList.Count ; i++ )
			{
				oLoopChassis = (Chassis)oTempList.GetByIndex( i );

				oHitTestResults = oLoopChassis.HitTest( oDamageWhere );
				//The chassis getting hit takes 2/3rds the damage.  If only one chassis is left alive then
                //we don't care who was hit, that one takes the full brunt.
				if( oLoopChassis.StructurePoints > 0 &&
                    (
                        nLiveChassisCount == 1 
                        ||
                        (   
                            bMainChassisHitFound == false && Region.IsEmpty( oHitTestResults ) == false 
                        )
                    )
                  )
				{
					bMainChassisHitFound = true;
                    if (nLiveChassisCount == 1)
                    {
                        nDamageNotUsed += oLoopChassis.DealDamage(oSession,nPostDamageRedux, oHitTestResults);
                    }
                    else
                    {
                        nDamageNotUsed += oLoopChassis.DealDamage(oSession, nPostDamageRedux * .5, oHitTestResults);
                    }
				}
				else
				{
					nTempDamage = nPostDamageRedux * .5;
                    if (nLiveChassisCount - 1 != 0)
					{
                        nTempDamage = nTempDamage / (nLiveChassisCount - 1);
					}
                    nDamageNotUsed += oLoopChassis.DealDamage(oSession, nTempDamage, oDamageWhere);
				}

				//Don't want to add in negatives, they'd take us below zero.
				if( oLoopChassis.StructurePoints > 0 )
				{
					nTotalStructureRemaining += oLoopChassis.StructurePoints;
				}
			}

			//When its all said and done we have to have a base chassis, but it may have been damaged and lost.
			if( m_oBaseChassis == null && m_oChasses.Count > 0 )
			{
				m_oBaseChassis = (Chassis)m_oChasses.GetByIndex( 0 );
			}

			//Am I dead?
            if (base.IsDead == false && (m_oChasses.Count == 0 || nTotalStructureRemaining <= 0))
			{
				base.IsDead = true;
                base.HandleEntityDeath(oSession, oEntityDealingDamage);
				//base.OwnerSocketID = -1;
				base.CanPickUpItems = false;
			}
            

			return( nDamageNotUsed );		
		}
		public override Region HitTest( Region oTestRegion )
		{
			Chassis oLoopChassis = null;
			ArrayList oRegions = new ArrayList();
			Region oHitTestResults = null;
			Region oRetVal = null;


			for( int i=0 ; i<m_oChasses.Count ; i++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( i );
				oHitTestResults = oLoopChassis.HitTest( oTestRegion );
				if( Region.IsEmpty( oHitTestResults ) == true )
				{
					oRetVal = oHitTestResults;
					break;
				}
			}


			return( oRetVal );
		}
		public override bool Collision( Session oSession,Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion )
		{
			Chassis oLoopChassis = null;
			ArrayList oRegions = new ArrayList();
			Region oHitTestResults = null;
			DSSortedList oTempList = null;


			base.Collision( oSession,oCollidedWith, nElapsedTime, oCollisionRegion );


			//Get a copy of our chassis so that if one is removed we are still ok
			lock( m_oChasses )
			{
				oTempList = m_oChasses.Clone();
			}

			for( int i=0 ; i<oTempList.Count ; i++ )
			{
				oLoopChassis = (Chassis)oTempList.GetByIndex( i );

				oHitTestResults = oLoopChassis.HitTest( oCollisionRegion );
				//The chassis getting hit 
				if( Region.IsEmpty( oHitTestResults ) == false )
				{					
					oLoopChassis.Collision( oCollidedWith,nElapsedTime,oCollisionRegion );
				}
			}


			return( true );
		}

		private double GetLevelSpecialTraitTotal( enumEntProperties nPropertyToGet )
		{
			ItemSpecialTrait oLoopItem = null;
			double nMyValue = 0;
 
			for( int i=0 ; i<this.LevelSpecialTraits.Count ; i++ )
			{
				oLoopItem = (ItemSpecialTrait)this.LevelSpecialTraits[ i ];
				if( oLoopItem.SpecialTraitEnum == nPropertyToGet )
				{
					nMyValue += oLoopItem.Value;
				}
			}

			return( nMyValue );
		}

		public override double CalculatePropertyTotal( enumEntProperties nPropertyToGet )
		{
			Chassis oLoopChassis = null;
			double nPropertyTotal = 0;
			bool bGetForChildren = true;


			if( nPropertyToGet == enumEntProperties.Structure_Current || 
				nPropertyToGet == enumEntProperties.Structure_Max )
			{
				bGetForChildren = false;
			}

			nPropertyTotal = base.CalculatePropertyTotal( nPropertyToGet );

			//We always get level 1 for now
			if( nPropertyToGet == enumEntProperties.HUD_HPDetail )
			{				
				nPropertyTotal = this.BaseHUDLevel;
			}

			for( int n=0 ; n<m_oChasses.Count ; n++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( n );

				if( nPropertyToGet == enumEntProperties.HUD_HPDetail )
				{
					nPropertyTotal = DSMisc.Max( nPropertyTotal,
												 oLoopChassis.GetPropertyTotal( nPropertyToGet,bGetForChildren ) );
				}
				else
				{
					nPropertyTotal += oLoopChassis.GetPropertyTotal( nPropertyToGet,bGetForChildren );
				}
			}

			nPropertyTotal += GetLevelSpecialTraitTotal( nPropertyToGet );


			return( nPropertyTotal );
		}
		public virtual double TakeEnergy( double nAmountToTake )
		{
			Chassis oLoopChassis = null;
			double nEnergyFound = 0;


			//Take from our produced energy first
			if( nAmountToTake <= m_nEnergyForRound )
			{
				nEnergyFound = nAmountToTake;
				m_nEnergyForRound -= nAmountToTake;
				nAmountToTake = 0;
			}
			else
			{
				nEnergyFound = m_nEnergyForRound;
				nAmountToTake -= m_nEnergyForRound;
				m_nEnergyForRound = 0;
			}

			if( nAmountToTake > 0 )
			{
				for( int n=0 ; n<m_oChasses.Count ; n++ )
				{
					oLoopChassis = (Chassis)m_oChasses.GetByIndex( n );

					nEnergyFound += oLoopChassis.TakeEnergy( nAmountToTake );
				}
			}

			return( nEnergyFound );
		}
		public virtual double GetEnergyAvailable()
		{
			return( GetPropertyTotal( enumEntProperties.Energy_Available ) );
		}
		public void RestoreToFullHealth()
		{
			Chassis oLoopChassis = null;

			this.StructurePoints = this.MaxStructurePoints;

			for( int n=0 ; n<m_oChasses.Count ; n++ )
			{
				oLoopChassis = (Chassis)m_oChasses.GetByIndex( n );
				oLoopChassis.RestoreToFullHealth();
			}
		}

		#region HUD rendering
		private void RenderHPBarHUD( RenderSettings oRenderSettings )
		{
			enumHUDDetail nHUDEnum = enumHUDDetail.Name;
			int nHUDLevel = 0;


			nHUDLevel = (int)CalculatePropertyTotal( enumEntProperties.HUD_HPDetail );
			nHUDEnum = (enumHUDDetail)nHUDLevel;

            foreach( Entity oLoopEntity in Globals.Inst().MyZone.Entitys.Values )
			{
				if( oLoopEntity != this && oLoopEntity.Name.Length > 0 &&
					( oLoopEntity.IsWaveEntity == true || oLoopEntity.EntityType == enumEntityType.Ship ) )
				{
					RenderEntityName( nHUDEnum,oRenderSettings,oLoopEntity );
				}
			}
		}
		private void RenderEntityName( enumHUDDetail nHUDLevel,RenderSettings oRenderSettings,Entity oEntity )
		{
			double nHPPerc = 0, nShieldPerc = 0, nEnergyPerc = 0;
			int nBarWidth = 95, nMainBarHeight = 10, nSubBarHeight = 6;
            int nSecondaryBarHeights = 6;
			int nHUDStartY = 0;
			int nColor = 0;
			System.Drawing.Color oColor = System.Drawing.Color.White;
			System.Drawing.Rectangle oTargetRect = System.Drawing.Rectangle.Empty;
			ComplexEntity oCEntity = null;
			string sName = "";
			Vector2 vScrPos = Vector2.Empty;


			#region Calculate our stat percentages
			//And what is their HP percent?
			if( Globals.ObjectIsComplexEntity( oEntity ) == true )
			{
				oCEntity = (ComplexEntity)oEntity;
				nHPPerc = oCEntity.GetPropertyTotal( enumEntProperties.Structure_Current ) / 
						  oCEntity.GetPropertyTotal( enumEntProperties.Structure_Max );
				nShieldPerc = oCEntity.GetPropertyTotal( enumEntProperties.Shield_Current ) / 
							  oCEntity.GetPropertyTotal( enumEntProperties.Shield_Max );
				nEnergyPerc = oCEntity.GetPropertyTotal( enumEntProperties.Energy_Available ) / 
							  oCEntity.GetPropertyTotal( enumEntProperties.Energy_MaxStorage );
			}
			else
			{
				nHPPerc = oEntity.StructurePoints / oEntity.MaxStructurePoints;
			}
            if (nHPPerc < 0) { nHPPerc = 0; }
            if (nShieldPerc < 0) { nShieldPerc = 0; }
            if (nEnergyPerc < 0) { nEnergyPerc = 0; }
			#endregion

			vScrPos = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,oEntity.Pos );
			nHUDStartY = (int)( vScrPos.Y + 25 );

			if( (int)nHUDLevel >= (int)enumHUDDetail.Name )
			{
				//Get our color and name
				sName = oEntity.Name;
				if( oEntity.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED )
				{
					oColor = System.Drawing.Color.LightGreen;
				}
				else if( oEntity.IsWaveEntity == true )
				{
					oColor = System.Drawing.Color.Red;
				}
			}

			if( nHUDLevel == enumHUDDetail.Name || nHUDLevel == enumHUDDetail.HP3ColorIndicator )
			{
				Globals.Inst().GameEngine.RenderText( 
						vScrPos.X - ( sName.Length / 1.0 ) * 2.5,nHUDStartY - 7,
						System.Drawing.Color.FromArgb( 150,oColor ),sName );
			}
			if( (int)nHUDLevel == (int)enumHUDDetail.HP3ColorIndicator )
			{
				if( nHPPerc < .25 )		{ nColor = System.Drawing.Color.Red.ToArgb(); }
				else if( nHPPerc < .5 )	{ nColor = System.Drawing.Color.Yellow.ToArgb(); }
				else					{ nColor = System.Drawing.Color.LightGreen.ToArgb(); }

				oTargetRect = new Rectangle(	(int)( vScrPos.X - ( sName.Length / 1.0 ) * 2.5 - 11 ),
												nHUDStartY - 4,8,8 );
				Globals.Inst().GameEngine.RenderTexture2D( "HP Indicator",
									System.Drawing.Rectangle.Empty,
									oTargetRect,Vector2.Empty,0,0,false,nColor );
			}
			if( (int)nHUDLevel >= (int)enumHUDDetail.HPBar )
			{
				RenderBarEx( (int)vScrPos.X,nHUDStartY,nBarWidth,
						nMainBarHeight,
						System.Drawing.Color.FromArgb( 100,0,0 ).ToArgb(),System.Drawing.Color.Red.ToArgb(),
						System.Drawing.Color.White.ToArgb(),nHPPerc,sName );
			}
			if( oCEntity != null )
			{
				if( (int)nHUDLevel >= (int)enumHUDDetail.HPShieldEnergy )
				{
                    RenderBar((int)vScrPos.X, nHUDStartY + (int)(nMainBarHeight / 2.0) + 3, nBarWidth, nSecondaryBarHeights,
						//System.Drawing.Color.FromArgb( 166,172,0 ).ToArgb(),
						System.Drawing.Color.FromArgb( (int)( 166*.8 ),(int)( 172*.8 ),0 ).ToArgb(),
						System.Drawing.Color.Yellow.ToArgb(),nEnergyPerc );
				}
				if( (int)nHUDLevel >= (int)enumHUDDetail.HPShieldBar )
				{
					RenderBar( (int)vScrPos.X,nHUDStartY + (int)( nMainBarHeight / 2.0 ) + (int)( nSubBarHeight / 2.0 ) + 3,
                        nBarWidth, nSecondaryBarHeights, System.Drawing.Color.FromArgb(0, 0, 140).ToArgb(),
						System.Drawing.Color.Blue.ToArgb(),nShieldPerc );
				}
			}
		}
		private void RenderBar( int nCenterX,int nCenterY,int nWidth,int nHeight,
								int nBaseColor,int nMainColor,double nPerc )
		{
			System.Drawing.Rectangle oTargetRect = System.Drawing.Rectangle.Empty;


			//Render the max size
			oTargetRect = new Rectangle(	
					(int)( nCenterX - nWidth / 2.0 ),
					(int)( nCenterY - nHeight / 2.0 ),
					nWidth, nHeight );
			Globals.Inst().GameEngine.RenderRect2D( oTargetRect,nBaseColor,0 );

			//Render their percent
			oTargetRect = new Rectangle(	
					(int)( nCenterX - nWidth / 2.0 + 1 ),
					(int)( nCenterY - nHeight / 2.0 + 1 ),
					(int)( ( nWidth - 2 ) * nPerc ),
					(int)( nHeight - 2 ) );
			Globals.Inst().GameEngine.RenderRect2D( oTargetRect,nMainColor,0 );
		}
		private void RenderBarEx( int nCenterX,int nCenterY,int nWidth,int nHeight,
								  int nBaseColor,int nMainColor,int nTextMainColor,
								  double nPerc,string sText )
		{
			System.Drawing.Rectangle oTargetRect = System.Drawing.Rectangle.Empty;
			System.Drawing.Rectangle oPercentBar = System.Drawing.Rectangle.Empty;
			double nTextX = 0, nTextY = 0;


			nTextX = nCenterX - ( sText.Length / 1.0 ) * 2.5;
			nTextY = nCenterY - 7;

			//Render the max size
			oTargetRect = new Rectangle(	
				(int)( nCenterX - nWidth / 2.0 ),
				(int)( nCenterY - nHeight / 2.0 ),
				nWidth, nHeight );
			Globals.Inst().GameEngine.RenderRect2D( oTargetRect,nBaseColor,0 );

			//Render their percent
			oPercentBar = new Rectangle(	
				(int)( nCenterX - nWidth / 2.0 + 1 ),
				(int)( nCenterY - nHeight / 2.0 + 1 ),
				(int)( ( nWidth - 2 ) * nPerc ),
				(int)( nHeight - 2 ) );
			Globals.Inst().GameEngine.RenderRect2D( oPercentBar,nMainColor,0 );

			//Render the main color text
			Globals.Inst().GameEngine.RenderText( nTextX,nTextY,
				System.Drawing.Color.FromArgb( nTextMainColor ),sText );
		}
		#endregion


		#region Properties
		public DSSortedList Chasses
		{
			get
			{
				return( m_oChasses );
			}
			set
			{
				m_oChasses = value;
			}
		}
		public Chassis BaseChassis
		{
			get
			{
				return( m_oBaseChassis );
			}
			set
			{
				m_oBaseChassis = value;
			}
		}
		public override double Mass
		{
			get
			{
				return( GetPropertyTotal( enumEntProperties.Structure_Mass ) );
			}
			set
			{
				base.Mass = value;
			}
		}
		/*public override long EXPValue
		{
			get
			{
				return( 0 );
			}
		}*/

		public override string Name
		{
			get
			{
				if( this.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED )
				{
                    m_sName = Globals.Inst().Session.GetPlayerName( this.OwnerSocketID );
				}
				return( m_sName );
			}
		}
		public int BaseHUDLevel
		{
			get
			{
				return( m_nBaseHUDLevel );
			}
			set
			{
				m_nBaseHUDLevel = value;
			}
		}
		#endregion
	}
}